import pygame
import random
import time
import argparse
import numpy as np
from typing import List, Tuple
from collections import deque

class MazeAnalyzer:
    """Analyzer class for computing metrics and statistics"""
    def __init__(self):
        self.stats = {
            'loops': 0,
            'visited_cells': 0,
            'backtracks': 0,
            'memory_used': 0,
            'path_length': 0,
            'explored_ratio': 0,
            'time_taken': 0
        }

    def analyze_solving_process(self):
        """Analyze and display solving metrics"""
        total_cells = self.size * self.size
        explored_ratio = len(self.explored) / total_cells * 100
        
        # Calculate memory usage (approximate)
        memory_explored = len(self.explored) * 8  # bytes for set storage
        memory_path = len(self.path) * 8         # bytes for path storage
        memory_queue = len(self.explored) * 16    # bytes for queue/priority queue
        total_memory = memory_explored + memory_path + memory_queue
        
        self.stats = {
            'loops': self.iterations,
            'visited_cells': len(self.explored),
            'backtracks': self.iterations - len(self.path),
            'memory_used': total_memory,
            'path_length': len(self.path),
            'explored_ratio': explored_ratio,
            'time_taken': self.end_time - self.start_time
        }
        
        # Print detailed analysis
        print("\n=== Maze Solving Analysis ===")
        
        print("\nPerformance Metrics:")
        print(f"Time taken: {self.stats['time_taken']:.3f} seconds")
        print(f"Total iterations: {self.stats['loops']}")
        print(f"Cells visited: {self.stats['visited_cells']} out of {total_cells}")
        print(f"Exploration coverage: {self.stats['explored_ratio']:.1f}%")
        print(f"Solution path length: {self.stats['path_length']}")
        print(f"Backtracking steps: {self.stats['backtracks']}")
        print(f"Memory used: {self.stats['memory_used']/1024:.2f} KB")
        
        print("\nComplexity Analysis:")
        print("Time Complexity: O(V + E)")  # For A* on a grid
        print("Space Complexity: O(V)")     # V = number of vertices/cells
        print("Where V = grid size² and E = possible moves between cells")
        
        print("\nAlgorithm Statistics:")
        print(f"Average steps per cell: {self.iterations/len(self.explored):.2f}")
        print(f"Solution efficiency: {(len(self.path)/len(self.explored))*100:.1f}%")
        print(f"Search efficiency: {(len(self.path)/self.iterations)*100:.1f}%")

        print("\nClassical Computing Characteristics:")
        print("1. Sequential cell exploration")
        print("2. Memory-based path tracking")
        print("3. Deterministic shortest path")
        print("4. Backtracking capability")
        
        # Optimization suggestions
        print("\nOptimization Opportunities:")
        if self.stats['explored_ratio'] > 70:
            print("- High exploration ratio: Consider using different heuristic")
        if self.stats['backtracks'] > len(self.path):
            print("- High backtracking: Consider bidirectional search")
        if self.stats['time_taken'] > 1.0:
            print("- Consider implementing parallel search patterns")

class ImprovedMazeGame(MazeAnalyzer):
    def __init__(self, size: int = 30, complexity: float = 0.7):
        super().__init__()
        pygame.init()
        
        # Screen setup
        self.size = size
        self.cell_size = 25
        self.wall_thickness = 2
        self.padding = 40
        
        self.width = size * self.cell_size + 2 * self.padding
        self.height = size * self.cell_size + 2 * self.padding
        
        self.screen = pygame.display.set_mode((self.width, self.height))
        pygame.display.set_caption("Maze Solver with Analysis")
        
        # Colors
        self.BACKGROUND = (255, 255, 255)
        self.WALL_COLOR = (70, 70, 70)
        self.START_COLOR = (50, 150, 255)  # Blue
        self.END_COLOR = (255, 50, 50)     # Red
        self.PATH_COLORS = self._generate_gradient_colors()
        
        # Game state
        self.maze = self.generate_maze(size, complexity)
        self.start = (1, 1)
        self.end = (size-2, size-2)
        self.current_pos = self.start
        self.path = []
        self.explored = set()
        self.moves = []
        
        # Statistics
        self.start_time = None
        self.end_time = None
        self.iterations = 0

    def _generate_gradient_colors(self, steps: int = 100) -> List[Tuple[int, int, int]]:
        """Generate gradient colors from start_color to end_color"""
        colors = []
        for i in range(steps):
            ratio = i / (steps-1)
            r = int(self.START_COLOR[0] * (1-ratio) + self.END_COLOR[0] * ratio)
            g = int(self.START_COLOR[1] * (1-ratio) + self.END_COLOR[1] * ratio)
            b = int(self.START_COLOR[2] * (1-ratio) + self.END_COLOR[2] * ratio)
            colors.append((r, g, b))
        return colors

    def generate_maze(self, size: int, complexity: float) -> List[List[int]]:
        """Generate a solvable maze"""
        # Initialize empty maze
        maze = [[0 for _ in range(size)] for _ in range(size)]
        
        # Add border walls
        for i in range(size):
            maze[0][i] = maze[size-1][i] = maze[i][0] = maze[i][size-1] = 1

        # Add random walls
        walls = int((size * size) * complexity * 0.3)
        for _ in range(walls):
            x = random.randint(1, size-2)
            y = random.randint(1, size-2)
            
            # Don't block start or end
            if (x, y) not in [(1,1), (size-2,size-2)]:
                maze[x][y] = 1
                
                # Check if maze is still solvable
                if not self.is_solvable(maze):
                    maze[x][y] = 0

        return maze

    def is_solvable(self, maze: List[List[int]]) -> bool:
        """Check if maze has a path from start to end"""
        visited = set()
        queue = deque([(1, 1)])  # Start position
        
        while queue:
            x, y = queue.popleft()
            if (x, y) == (self.size-2, self.size-2):  # End position
                return True
                
            for dx, dy in [(0,1), (1,0), (0,-1), (-1,0)]:
                new_x, new_y = x + dx, y + dy
                if (0 <= new_x < self.size and 
                    0 <= new_y < self.size and 
                    maze[new_x][new_y] == 0 and 
                    (new_x, new_y) not in visited):
                    visited.add((new_x, new_y))
                    queue.append((new_x, new_y))
        
        return False

    def solve_maze(self):
        """Solve maze using A* algorithm"""
        self.start_time = time.time()
        self.explored = set()
        self.path = []
        self.iterations = 0
        
        def heuristic(pos):
            return abs(pos[0] - self.end[0]) + abs(pos[1] - self.end[1])
        
        # A* algorithm
        open_set = {self.start}
        came_from = {}
        g_score = {self.start: 0}
        f_score = {self.start: heuristic(self.start)}
        
        while open_set:
            current = min(open_set, key=lambda x: f_score.get(x, float('inf')))
            
            if current == self.end:
                # Reconstruct path
                while current in came_from:
                    self.path.append(current)
                    current = came_from[current]
                self.path.append(self.start)
                self.path.reverse()
                break
            
            open_set.remove(current)
            self.explored.add(current)
            self.iterations += 1
            
            for dx, dy in [(0,1), (1,0), (0,-1), (-1,0)]:
                next_x, next_y = current[0] + dx, current[1] + dy
                next_pos = (next_x, next_y)
                
                if (0 <= next_x < self.size and 
                    0 <= next_y < self.size and 
                    self.maze[next_x][next_y] == 0):
                    
                    tentative_g_score = g_score[current] + 1
                    
                    if tentative_g_score < g_score.get(next_pos, float('inf')):
                        came_from[next_pos] = current
                        g_score[next_pos] = tentative_g_score
                        f_score[next_pos] = tentative_g_score + heuristic(next_pos)
                        open_set.add(next_pos)
                        
                        self.current_pos = next_pos
                        self.moves.append(f"Move to {next_pos}")
                        self.draw_maze()
                        pygame.time.delay(50)
        
        self.end_time = time.time()

    def draw_maze(self):
        """Draw the maze and solution path"""
        self.screen.fill(self.BACKGROUND)
        
        # Draw walls
        for i in range(self.size):
            for j in range(self.size):
                if self.maze[i][j] == 1:
                    x = j * self.cell_size + self.padding
                    y = i * self.cell_size + self.padding
                    pygame.draw.rect(self.screen, 
                                   self.WALL_COLOR,
                                   (x, y, self.cell_size, self.cell_size),
                                   self.wall_thickness)
        
        # Draw explored cells
        for x, y in self.explored:
            pygame.draw.rect(self.screen,
                           (240, 240, 240),
                           (y * self.cell_size + self.padding,
                            x * self.cell_size + self.padding,
                            self.cell_size,
                            self.cell_size))
        
        # Draw path
        if len(self.path) > 1:
            for i in range(len(self.path)-1):
                progress = i / (len(self.path)-1)
                color_idx = int(progress * (len(self.PATH_COLORS)-1))
                color = self.PATH_COLORS[color_idx]
                
                start_x = self.path[i][1] * self.cell_size + self.padding + self.cell_size//2
                start_y = self.path[i][0] * self.cell_size + self.padding + self.cell_size//2
                end_x = self.path[i+1][1] * self.cell_size + self.padding + self.cell_size//2
                end_y = self.path[i+1][0] * self.cell_size + self.padding + self.cell_size//2
                
                pygame.draw.line(self.screen, color,
                               (start_x, start_y),
                               (end_x, end_y),
                               self.wall_thickness + 1)
        
        # Draw start and end points
        pygame.draw.circle(self.screen, self.START_COLOR,
                         (self.start[1] * self.cell_size + self.padding + self.cell_size//2,
                          self.start[0] * self.cell_size + self.padding + self.cell_size//2),
                         self.cell_size//3)
        
        pygame.draw.circle(self.screen, self.END_COLOR,
                         (self.end[1] * self.cell_size + self.padding + self.cell_size//2,
                          self.end[0] * self.cell_size + self.padding + self.cell_size//2),
                         self.cell_size//3)
        
        pygame.display.flip()

def main():
    parser = argparse.ArgumentParser(description='Maze Solver with Analysis')
    parser.add_argument('--size', type=int, default=30,
                       help='Size of the maze (default: 30)')
    parser.add_argument('--complexity', type=float, default=0.7,
                       help='Complexity of the maze (0.0-1.0, default: 0.7)')
    parser.add_argument('--show-analysis', action='store_true',
                       help='Show detailed analysis after solving')
    args = parser.parse_args()

    game = ImprovedMazeGame(args.size, args.complexity)
    running = True
    solved = False

    while running:
        for event in pygame.event.get():
            if event.type == pygame.QUIT:
                running = False
            elif event.type == pygame.KEYDOWN:
                if event.key == pygame.K_SPACE and not solved:
                    game.solve_maze()
                    if args.show_analysis:
                        game.analyze_solving_process()
                    solved = True
                elif event.key == pygame.K_r:  # Reset maze
                    game = ImprovedMazeGame(args.size, args.complexity)
                    solved = False

        game.draw_maze()
        pygame.time.delay(30)

    pygame.quit()

if __name__ == "__main__":
    main()